<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Role;
// use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use App\Models\RoleDataScope;
use App\Models\Department;

class RoleController extends Controller
{

private function permissionActionMap(): array
{
    return [
        'view'    => ['index', 'list', 'view'],
        'add'     => ['add', 'create', 'store'],
        'edit'    => ['edit', 'update'],
        'delete'  => ['delete', 'destroy'],
        'approve' => ['status', 'toggle', 'approve'],
    ];
}

private function buildPermissionMatrix()
{
    $permissions = Permission::pluck('name')->toArray();
    $actionMap = $this->permissionActionMap();

    $matrix = [];

    foreach ($permissions as $perm) {

        // Normalize name
        $parts = preg_split('/[.\-_]/', $perm);

        $module = $parts[0]; // brand, product, banner, users

        foreach ($actionMap as $action => $keywords) {
            foreach ($keywords as $keyword) {
                if (str_contains($perm, $keyword)) {
                    $matrix[$module][$action] = $perm;
                    break 2;
                }
            }
        }
    }

    return $matrix;
}



    public function create()
    {
        // Predefined modules & permissions
        $modules = [
            'profile' => ['profile.edit', 'profile.update', 'profile.destroy'],
            'room' => ['room.index', 'add-room', 'update-room', 'delete-room'],
            'tag' => ['tag.index', 'add-tag', 'update-tag', 'delete-tag'],
            'category' => ['category.index', 'add-category', 'update-category', 'delete-category'],
            'subcategory' => ['subcategory', 'add-subcategory', 'update-subcategory', 'delete-subcategory', 'toggle-subcategory-status'],
            'product' => ['products', 'edit-product', 'add-product', 'delete-product'],
            'banner' => ['banner', 'get-banner', 'add-banner', 'update-banner', 'banner.status'],
            'coupon' => ['coupons', 'get-coupons', 'add-coupon', 'coupon.update', 'coupon.status'],
            'stock' => ['stock.index', 'stock.list', 'stock.view', 'stock.edit'],
            'location' => ['locations', 'get-locations', 'add-locations', 'update-locations', 'delete-locations'],
            'roles' => ['roles.index', 'roles.create', 'roles.edit', 'roles.update', 'roles.destroy'],
            'users' => ['users.index', 'users.store', 'users.edit', 'users.update', 'users.destroy']
        ];

        // Fetch all permissions
        $allPermissions = Permission::all()->pluck('name')->toArray();

        // Group permissions by module
        $permissions = [];
        foreach ($modules as $module => $perms) {
            $permissions[$module] = array_values(array_intersect($allPermissions, $perms));
        }

        $departments = Department::all();

        return view('admin.roles.create', compact('permissions', 'departments'));
    }


    public function store(Request $request)
    {
        try {
            $request->validate([
                'role_name'       => 'required|string|unique:roles,name',
                'permissions'     => 'array',
                'department_ids'  => 'array',
                'department_ids.*' => 'exists:departments,id',
            ]);

            DB::transaction(function () use ($request) {

                /** -----------------------------
                 * 1️⃣ Create Role
                 * ---------------------------- */
                $role = Role::create([
                    'name'       => $request->role_name,
                    'guard_name' => 'web',
                ]);
                $adminId = User::first()?->id;
                if ($adminId) {
                    event(new \App\Events\SystemEvent([
                        'type' => 'role_created',
                        'subject' => $role,
                        'message' => "New Role {$role->name} created",
                        'recipient_id' => $adminId,
                        'channels' => ['dashboard'],
                    ]));
                }
                /** -----------------------------
                 * 2️⃣ Attach Permissions
                 * ---------------------------- */
                if ($request->filled('permissions')) {
                    $role->syncPermissions($request->permissions);
                }

                /** -----------------------------
                 * 3️⃣ Attach Departments (Pivot)
                 * ---------------------------- */
                if ($request->filled('department_ids')) {
                    $role->departments()->sync($request->department_ids);
                }


                /** -----------------------------
                 * 3️⃣ Save Role Data Scopes
                 * ---------------------------- */
                if ($request->filled('scopes')) {

                    foreach ($request->scopes as $module => $scope) {

                        // Skip invalid scope definitions
                        if (empty($scope['scope_type'])) {
                            continue;
                        }

                        RoleDataScope::create([
                            'role_id'    => $role->id,
                            'module'     => $module,
                            'scope_type' => $scope['scope_type'],

                            // JSON storage for dynamic values
                            'scope_value' => isset($scope['scope_value'])
                                ? json_encode($scope['scope_value'])
                                : null,

                            // System role lock (optional checkbox)
                            'is_locked' => $request->boolean('is_locked'),
                        ]);
                    }
                }
            });
            // Notify Admin

            return redirect()
                ->route('roles.index')
                ->with('success', 'Role created successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Redirect back with validation errors
            return redirect()
                ->back()
                ->withErrors($e->errors())  // Laravel default validation errors
                ->withInput();             // keep old input

        } catch (\Exception $e) {
            // Redirect back with a generic error
            return redirect()
                ->back()
                ->with('error', 'Something went wrong: ' . $e->getMessage())
                ->withInput();
        }
    }



    public function index()
    {
        // Get all roles with permissions
        $roles = Role::with('permissions')->get();

        // Group permissions per role by module
        $roles->each(function ($role) {
            $grouped = [];

            foreach ($role->permissions as $permission) {
                $name = strtolower($permission->name);

                // Detect module
                if (str_contains($name, 'subcategory')) {
                    $grouped['Sub Category'][] = $permission->name;
                } elseif (str_contains($name, 'category')) {
                    $grouped['Category'][] = $permission->name;
                } elseif (str_contains($name, 'profile')) {
                    $grouped['Profile'][] = $permission->name;
                } elseif (str_contains($name, 'room')) {
                    $grouped['Room'][] = $permission->name;
                } elseif (str_contains($name, 'tag')) {
                    $grouped['Tag'][] = $permission->name;
                } elseif (str_contains($name, 'product')) {
                    $grouped['Product'][] = $permission->name;
                } elseif (str_contains($name, 'banner')) {
                    $grouped['Banner'][] = $permission->name;
                } elseif (str_contains($name, 'coupon')) {
                    $grouped['Coupon'][] = $permission->name;
                } elseif (str_contains($name, 'stock')) {
                    $grouped['Stock'][] = $permission->name;
                } elseif (str_contains($name, 'location')) {
                    $grouped['Location'][] = $permission->name;
                } elseif (str_contains($name, 'role')) {
                    $grouped['Roles'][] = $permission->name;
                } elseif (str_contains($name, 'user')) {
                    $grouped['Users'][] = $permission->name;
                } else {
                    $grouped['Other'][] = $permission->name;
                }
            }

            $role->grouped_permissions = $grouped;
        });

        return view('admin.roles.index', compact('roles'));
    }




    public function destroy(Role $role)
    {
        if ($role->name === 'Super Admin') {
            return response()->json([
                'approve' => false,
                'message' => 'Admin role cannot be deleted'
            ], 403);
        }

        $role->delete();

        return response()->json([
            'approve' => true,
            'message' => 'Role deleted successfully'
        ]);
    }

    public function edit(Role $role)
    {
        // -----------------------------
        // Modules & permissions (UNCHANGED)
        // -----------------------------
        $modules = [
            'profile' => [
                'view'   => 'profile.index',
                'edit'   => 'profile.edit',
                'delete' => 'profile.destroy',
            ],
            'room' => [
                'view'   => 'room.index',
                'add'    => 'add-room',
                'edit'   => 'update-room',
                'delete' => 'delete-room',
            ],
            'tag' => [
                'view'   => 'tag.index',
                'add'    => 'add-tag',
                'edit'   => 'update-tag',
                'delete' => 'delete-tag',
            ],
            'category' => [
                'view'   => 'category.index',
                'add'    => 'add-category',
                'edit'   => 'update-category',
                'delete' => 'delete-category',
            ],
            'subcategory' => [
                'view'   => 'subcategory',
                'add'    => 'add-subcategory',
                'edit'   => 'update-subcategory',
                'delete' => 'delete-subcategory',
                'approve' => 'toggle-subcategory-status'
            ],
            'product' => [
                'view'   => 'products',
                'add'    => 'add-product',
                'edit'   => 'edit-product',
                'delete' => 'delete-product'
            ],
            'banner' => [
                'view'   => 'banner',
                'add'    => 'add-banner',
                'edit'   => 'update-banner',
                'delete' => 'banner-delete',
                'approve' => 'banner.status'
            ],
            'coupon' => [
                'view'   => 'coupons',
                'add'    => 'add-coupon',
                'edit'   => 'edit-coupon',
                'delete' => 'coupon-delete',
                'approve' => 'coupon.status'
            ],
            'stock' => [
                'view' => 'stock.index',
                'list' => 'stock.list',
                'edit' => 'stock.edit'
            ],
            'location' => [
                'view'   => 'locations',
                'add'    => 'add-locations',
                'edit'   => 'edit-location',
                'delete' => 'delete-locations'
            ],
            'roles' => [
                'view'   => 'roles.index',
                'add'    => 'roles.create',
                'edit'   => 'roles.edit',
                'delete' => 'roles.destroy'
            ],
            'users' => [
                'view'   => 'users.index',
                'add'    => 'users.store',
                'edit'   => 'users.edit',
                'delete' => 'users.destroy'
            ],
        ];

        // -----------------------------
        // Build permission matrix
        // -----------------------------
        $permissions = [];
        $allPermissions = Permission::pluck('name')->toArray();

        foreach ($modules as $module => $actions) {
            foreach ($actions as $action => $permName) {
                if (in_array($permName, $allPermissions)) {
                    $permissions[$module][$action] = $permName;
                }
            }
        }

        // -----------------------------
        // Role current data
        // -----------------------------
        $rolePermissions = $role->permissions->pluck('name')->toArray();

        $departments = Department::where('is_active', 1)->get();
        $selectedDepartments = $role->departments->pluck('id')->toArray();
        $roleDepartmentIds = $selectedDepartments;
        $roleScopes = [];

        foreach ($role->dataScopes as $scope) {
            $roleScopes[$scope->module] = [
                'scope_type'  => strtolower($scope->scope_type),
                'scope_value' => $scope->scope_value
                    ? json_decode($scope->scope_value, true)
                    : []
            ];
        }
        //print_r($permissions);die();
        return view('admin.roles.edit', compact(
            'role',
            'permissions',
            'rolePermissions',
            'departments',
            'selectedDepartments',
            'roleDepartmentIds',
            'roleScopes'
        ));
    }



    public function update(Request $request, Role $role)
    {
        try {
            $request->validate([
                'role_name'        => 'required|string|unique:roles,name,' . $role->id,
                'permissions'      => 'array',
                'permissions.*'    => 'exists:permissions,name',
                'department_ids'   => 'array',
                'department_ids.*' => 'exists:departments,id',
                'scopes'           => 'array',
            ]);

            DB::transaction(function () use ($request, $role) {

                /** -----------------------------
                 * 1️⃣ Update Role
                 * ---------------------------- */
                $role->update([
                    'name' => $request->role_name,
                ]);

                /** -----------------------------
                 * 2️⃣ Sync Permissions
                 * ---------------------------- */
                $role->syncPermissions($request->permissions ?? []);

                /** -----------------------------
                 * 3️⃣ Sync Departments (Pivot)
                 * ---------------------------- */
                if ($request->has('department_ids')) {
                    $role->departments()->sync($request->department_ids);
                } else {
                    // If none selected, remove all
                    $role->departments()->detach();
                }

                /** -----------------------------
                 * 4️⃣ Save / Update Role Data Scopes
                 * ---------------------------- */
                // Remove old scopes (edit mode)
                RoleDataScope::where('role_id', $role->id)->delete();

                if ($request->filled('scopes')) {

                    foreach ($request->scopes as $module => $scope) {

                        // Skip invalid scope definitions
                        if (empty($scope['scope_type'])) {
                            continue;
                        }

                        // Normalize scope_value
                        $scopeValue = null;

                        if (!empty($scope['scope_value'])) {
                            $scopeValue = is_array($scope['scope_value'])
                                ? json_encode(array_values($scope['scope_value']))
                                : json_encode([$scope['scope_value']]);
                        }

                        RoleDataScope::create([
                            'role_id'     => $role->id,
                            'module'      => $module,
                            'scope_type'  => strtoupper($scope['scope_type']),
                            'scope_value' => $scopeValue,
                            'is_locked'   => $request->boolean('is_locked'),
                        ]);
                    }
                }
            });
            $adminId = User::first()?->id;
            if ($adminId) {
                event(new \App\Events\SystemEvent([
                    'type' => 'role_updated',
                    'subject' => $role,
                    'message' => "New Role {$role->name} upted",
                    'recipient_id' => $adminId,
                    'channels' => ['dashboard'],
                ]));
            }
            return response()->json([
                'approve' => true,
                'message' => 'Role updated successfully'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Redirect back with validation errors
            return redirect()
                ->back()
                ->withErrors($e->errors())  // Laravel default validation errors
                ->withInput();             // keep old input

        } catch (\Exception $e) {
            // Redirect back with a generic error
            return redirect()
                ->back()
                ->with('error', 'Something went wrong: ' . $e->getMessage())
                ->withInput();
        }
    }






    public function deleteSelectedPermissions(Request $request)
    {
        // Predefined list of permissions that are protected (cannot be deleted)
        $permissionsToDelete  = [
            'products.step.one',
            'product.step2.store',
            'product.images.store',
            'stocks.store',
            'products.step.one.update',
            'products.step.two.update',
            'products.step.three.update',
            'products.step.four.update',
            'products.image.delete',
        ];

        DB::transaction(function () use ($permissionsToDelete) {
            foreach ($permissionsToDelete as $permName) {
                $permission = Permission::where('name', $permName)->first();
                if ($permission) {
                    // Remove permission from roles
                    foreach ($permission->roles as $role) {
                        $role->revokePermissionTo($permission);
                    }

                    // Remove permission from users
                    foreach ($permission->users as $user) {
                        $user->revokePermissionTo($permission);
                    }

                    // Delete the permission
                    $permission->delete();
                }
            }
        });

        return response()->json([
            'approve' => true,
            'message' => 'Selected permissions deleted successfully.'
        ]);
    }
}
